import { E as ElMessage } from "./element-plus.js";
(function polyfill() {
  const relList = document.createElement("link").relList;
  if (relList && relList.supports && relList.supports("modulepreload")) {
    return;
  }
  for (const link of document.querySelectorAll('link[rel="modulepreload"]')) {
    processPreload(link);
  }
  new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type !== "childList") {
        continue;
      }
      for (const node of mutation.addedNodes) {
        if (node.tagName === "LINK" && node.rel === "modulepreload")
          processPreload(node);
      }
    }
  }).observe(document, { childList: true, subtree: true });
  function getFetchOpts(link) {
    const fetchOpts = {};
    if (link.integrity) fetchOpts.integrity = link.integrity;
    if (link.referrerPolicy) fetchOpts.referrerPolicy = link.referrerPolicy;
    if (link.crossOrigin === "use-credentials")
      fetchOpts.credentials = "include";
    else if (link.crossOrigin === "anonymous") fetchOpts.credentials = "omit";
    else fetchOpts.credentials = "same-origin";
    return fetchOpts;
  }
  function processPreload(link) {
    if (link.ep)
      return;
    link.ep = true;
    const fetchOpts = getFetchOpts(link);
    fetch(link.href, fetchOpts);
  }
})();
const baseUrl = "http://localhost:8002";
const wsUrl = "ws://localhost:8002";
const request = async (url, options = {}) => {
  const token = await new Promise((resolve) => {
    if (typeof chrome !== "undefined" && chrome.storage && chrome.storage.local) {
      chrome.storage.local.get(["token"], (result) => {
        resolve(result.token || "");
      });
    } else {
      resolve("");
    }
  });
  const headers = {
    "Content-Type": "application/json",
    ...options.headers
  };
  if (token) {
    headers["Authorization"] = token;
  }
  const timeout = options.timeout || 9e4;
  const controller = new AbortController();
  const id = setTimeout(() => controller.abort(), timeout);
  const fetchOptions = {
    ...options,
    headers,
    signal: controller.signal
  };
  let fetchUrl = `${baseUrl}${url}`;
  if (options.method === "GET" && options.params) {
    const params = new URLSearchParams();
    Object.keys(options.params).forEach((key) => {
      if (options.params[key] !== void 0 && options.params[key] !== null && options.params[key] !== "") {
        params.append(key, options.params[key]);
      }
    });
    const queryString = params.toString();
    if (queryString) {
      fetchUrl += (fetchUrl.includes("?") ? "&" : "?") + queryString;
    }
    delete options.params;
  }
  try {
    const response = await fetch(fetchUrl, fetchOptions);
    clearTimeout(id);
    const data = await response.json();
    console.log("响应数据：", data);
    if (response.status === 401 || data.code === 401) {
      console.warn("登录失效，请重新登录");
      if (typeof chrome !== "undefined" && chrome.storage && chrome.storage.local) {
        chrome.storage.local.remove(["token", "userInfo"], () => {
          window.dispatchEvent(new CustomEvent("auth-expired"));
        });
      }
      return Promise.reject(data);
    }
    if (data.code !== 200 && data.code !== 0) {
      ElMessage.error(data.msg);
      return Promise.reject(data);
    }
    return data;
  } catch (error) {
    clearTimeout(id);
    console.error("API Request Error:", error);
    if (error.name === "AbortError") {
      const timeoutMsg = "请求超时，服务器响应过慢，请稍后重试";
      ElMessage.error(timeoutMsg);
      return Promise.reject({ code: 500, msg: timeoutMsg });
    }
    return Promise.reject(error);
  }
};
const http = {
  get: (url, params, options) => request(url, { ...options, params, method: "GET" }),
  post: (url, data, options) => request(url, { ...options, method: "POST", body: JSON.stringify(data) }),
  put: (url, data, options) => request(url, { ...options, method: "PUT", body: JSON.stringify(data) }),
  delete: (url, options) => request(url, { ...options, method: "DELETE" })
};
const _export_sfc = (sfc, props) => {
  const target = sfc.__vccOpts || sfc;
  for (const [key, val] of props) {
    target[key] = val;
  }
  return target;
};
export {
  _export_sfc as _,
  baseUrl as b,
  http as h,
  wsUrl as w
};
